#include <Arduino.h>
#include <WiFi.h>
#include <WebServer.h>
#include <ArduinoJson.h>
#include <math.h>

// Настройки WiFi
#define WIFI_SSID "ESP32-ADC-Spectrum"
#define WIFI_PASSWORD "12345678"
#define WEB_SERVER_PORT 80

// Настройки АЦП для ESP32-C3
#define ADC_PIN 0                 // GPIO0 - АЦП вход на ESP32-C3
#define SAMPLE_RATE 5000          // 5 kHz
#define BUFFER_SIZE 512           // Размер буфера для FFT
#define SPECTRUM_BINS 256         // Количество бинов спектра

// Пин светодиода для ESP32-C3
#define LED_PIN 2

WebServer server(WEB_SERVER_PORT);

// Глобальные переменные
volatile uint16_t adcBuffer[BUFFER_SIZE];
volatile bool bufferReady = false;
volatile uint32_t sampleIndex = 0;
float spectrum[SPECTRUM_BINS];
float magnitudeSpectrum[SPECTRUM_BINS/2];
uint16_t updateInterval = 500;   // Интервал обновления в миллисекундах

// БЫСТРЫЙ АЦП через таймер
hw_timer_t *timer = NULL;
portMUX_TYPE timerMux = portMUX_INITIALIZER_UNLOCKED;

void IRAM_ATTR onTimer() {
    portENTER_CRITICAL_ISR(&timerMux);
    if (sampleIndex < BUFFER_SIZE) {
        adcBuffer[sampleIndex] = analogRead(ADC_PIN);
        sampleIndex++;
    } else {
        bufferReady = true;
    }
    portEXIT_CRITICAL_ISR(&timerMux);
}

// Функция БПФ (упрощенная версия - DFT)
void computeFFT(float* input, float* output, uint16_t n) {
    // Дискретное преобразование Фурье
    for (int k = 0; k < n/2; k++) {
        float real = 0.0;
        float imag = 0.0;

        for (int i = 0; i < n; i++) {
            float angle = 2 * PI * k * i / n;
            real += input[i] * cos(angle);
            imag += input[i] * sin(angle);
        }

        output[k] = sqrt(real*real + imag*imag) / n;
    }
}

// Функция Хэмминга
void applyHammingWindow(float* data, uint16_t n) {
    for (int i = 0; i < n; i++) {
        data[i] = data[i] * (0.54 - 0.46 * cos(2 * PI * i / (n - 1)));
    }
}

// Вычисление автоспектра
void computeAutoSpectrum() {
    static float inputBuffer[BUFFER_SIZE];

    // Копируем данные из ADC буфера
    portENTER_CRITICAL(&timerMux);
    for (int i = 0; i < BUFFER_SIZE; i++) {
        // Нормализация -1 до 1 (12-bit ADC: 0-4095)
        inputBuffer[i] = (adcBuffer[i] - 2048.0) / 2048.0;
    }
    bufferReady = false;
    sampleIndex = 0;
    portEXIT_CRITICAL(&timerMux);

    // Применяем окно Хэмминга
    applyHammingWindow(inputBuffer, BUFFER_SIZE);

    // Вычисляем FFT
    computeFFT(inputBuffer, magnitudeSpectrum, BUFFER_SIZE);

    // Преобразуем в логарифмическую шкалу (дБ)
    for (int i = 0; i < SPECTRUM_BINS/2; i++) {
        float magnitude = magnitudeSpectrum[i];
        if (magnitude < 1e-10) magnitude = 1e-10;
        spectrum[i] = 20 * log10(magnitude);
    }
}

// Встроенные HTML страницы
const char INDEX_HTML[] PROGMEM = R"rawliteral(
<!DOCTYPE html>
<html>
<head>
    <title>ESP32-C3 Spectrum Analyzer</title>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 20px;
            background: #1e1e1e;
            color: #ffffff;
        }
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: #2d2d2d;
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.3);
        }
        .header {
            text-align: center;
            margin-bottom: 30px;
        }
        .controls {
            display: flex;
            gap: 15px;
            margin-bottom: 20px;
            flex-wrap: wrap;
            background: #3d3d3d;
            padding: 15px;
            border-radius: 8px;
        }
        .controls button {
            padding: 12px 24px;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            font-weight: bold;
            transition: all 0.3s;
        }
        .btn-start { background: #28a745; color: white; }
        .btn-stop { background: #dc3545; color: white; }
        .btn-clear { background: #ffc107; color: black; }
        .status-panel {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 20px;
        }
        .status-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            padding: 15px;
            border-radius: 8px;
            text-align: center;
        }
        .status-value {
            font-size: 24px;
            font-weight: bold;
            margin: 5px 0;
        }
        .status-label {
            font-size: 12px;
            opacity: 0.9;
        }
        .chart-container {
            position: relative;
            height: 400px;
            margin-bottom: 30px;
            background: #2d2d2d;
            padding: 20px;
            border-radius: 8px;
            border: 1px solid #444;
        }
        .chart-title {
            text-align: center;
            margin-bottom: 15px;
            font-size: 18px;
            font-weight: bold;
            color: #fff;
        }
        .frequency-controls {
            display: flex;
            gap: 15px;
            align-items: center;
            margin-bottom: 15px;
            flex-wrap: wrap;
        }
        .frequency-controls select {
            padding: 8px 12px;
            border: 1px solid #555;
            border-radius: 5px;
            background: #3d3d3d;
            color: #fff;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>📊 Анализатор спектра АЦП - ESP32-C3</h1>
            <p>АЦП пин: GPIO0 | Диапазон: 0-3.3V | Разрешение: 12 бит</p>
        </div>

        <div class="status-panel">
            <div class="status-card">
                <div class="status-label">Частота дискретизации</div>
                <div class="status-value" id="sampleRate">0</div>
                <div class="status-label">Гц</div>
            </div>
            <div class="status-card">
                <div class="status-label">Размер буфера</div>
                <div class="status-value" id="bufferSize">0</div>
                <div class="status-label">сэмплов</div>
            </div>
            <div class="status-card">
                <div class="status-label">Частота обновления</div>
                <div class="status-value" id="updateRate">0</div>
                <div class="status-label">Гц</div>
            </div>
            <div class="status-card">
                <div class="status-label">Макс. амплитуда</div>
                <div class="status-value" id="maxAmplitude">0</div>
                <div class="status-label">дБ</div>
            </div>
        </div>

        <div class="controls">
            <button class="btn-start" onclick="startAcquisition()">▶️ Запуск измерений</button>
            <button class="btn-stop" onclick="stopAcquisition()">⏹️ Стоп измерений</button>
            <button class="btn-clear" onclick="clearSpectrum()">🗑️ Очистить графики</button>
            <span id="acquisitionStatus" style="margin-left: auto; font-weight: bold; color: #dc3545;">Остановлено</span>
        </div>

        <div class="frequency-controls">
            <label>Частота дискретизации:</label>
            <select id="sampleRateSelect" onchange="changeSampleRate()">
                <option value="1000">1 kHz</option>
                <option value="2000">2 kHz</option>
                <option value="5000" selected>5 kHz</option>
                <option value="10000">10 kHz</option>
            </select>

            <label>Масштаб по Y:</label>
            <select id="scaleSelect" onchange="changeScale()">
                <option value="linear">Линейный</option>
                <option value="logarithmic" selected>Логарифмический</option>
            </select>
        </div>

        <div class="chart-container">
            <div class="chart-title">Автоспектр входного сигнала</div>
            <canvas id="spectrumChart"></canvas>
        </div>

        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px;">
            <div class="chart-container">
                <div class="chart-title">Сигнал АЦП (первые 200 точек)</div>
                <canvas id="timeDomainChart"></canvas>
            </div>
            <div class="chart-container">
                <div class="chart-title">Спектрограмма</div>
                <canvas id="spectrogramChart"></canvas>
            </div>
        </div>
    </div>

    <script>
        let spectrumChart, timeDomainChart, spectrogramChart;
        let isAcquiring = false;
        let updateInterval = 500; // 2 Hz update rate
        let spectrogramData = [];
        const maxSpectrogramPoints = 50;

        // Инициализация графиков
        function initializeCharts() {
            // График спектра
            const spectrumCtx = document.getElementById('spectrumChart').getContext('2d');
            spectrumChart = new Chart(spectrumCtx, {
                type: 'line',
                data: {
                    datasets: [{
                        label: 'Амплитуда спектра',
                        data: [],
                        borderColor: '#00ff88',
                        backgroundColor: 'rgba(0, 255, 136, 0.1)',
                        borderWidth: 2,
                        fill: true,
                        pointRadius: 0
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    animation: { duration: 0 },
                    scales: {
                        x: {
                            type: 'linear',
                            title: {
                                display: true,
                                text: 'Частота (Гц)',
                                color: '#fff'
                            },
                            grid: { color: '#444' },
                            ticks: { color: '#fff' }
                        },
                        y: {
                            type: 'linear',
                            title: {
                                display: true,
                                text: 'Амплитуда (дБ)',
                                color: '#fff'
                            },
                            grid: { color: '#444' },
                            ticks: { color: '#fff' }
                        }
                    },
                    plugins: {
                        legend: { labels: { color: '#fff' } },
                        tooltip: { mode: 'index', intersect: false }
                    }
                }
            });

            // График временной области
            const timeCtx = document.getElementById('timeDomainChart').getContext('2d');
            timeDomainChart = new Chart(timeCtx, {
                type: 'line',
                data: {
                    datasets: [{
                        label: 'Сигнал АЦП',
                        data: [],
                        borderColor: '#ff4444',
                        backgroundColor: 'rgba(255, 68, 68, 0.1)',
                        borderWidth: 1,
                        fill: true,
                        pointRadius: 0
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    animation: { duration: 0 },
                    scales: {
                        x: {
                            title: {
                                display: true,
                                text: 'Время (сэмплы)',
                                color: '#fff'
                            },
                            grid: { color: '#444' },
                            ticks: { color: '#fff' }
                        },
                        y: {
                            title: {
                                display: true,
                                text: 'АЦП значение',
                                color: '#fff'
                            },
                            grid: { color: '#444' },
                            ticks: { color: '#fff' }
                        }
                    },
                    plugins: {
                        legend: { labels: { color: '#fff' } }
                    }
                }
            });

            // Спектрограмма
            const spectrogramCtx = document.getElementById('spectrogramChart').getContext('2d');
            spectrogramChart = new Chart(spectrogramCtx, {
                type: 'line',
                data: { datasets: [] },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    animation: { duration: 0 },
                    scales: {
                        x: {
                            title: {
                                display: true,
                                text: 'Время',
                                color: '#fff'
                            },
                            grid: { color: '#444' },
                            ticks: { color: '#fff' }
                        },
                        y: {
                            title: {
                                display: true,
                                text: 'Частота (бин)',
                                color: '#fff'
                            },
                            grid: { color: '#444' },
                            ticks: { color: '#fff' }
                        }
                    },
                    plugins: {
                        legend: { display: false }
                    }
                }
            });
        }

        // Управление приобретением данных
        function startAcquisition() {
            isAcquiring = true;
            document.getElementById('acquisitionStatus').textContent = 'Активно';
            document.getElementById('acquisitionStatus').style.color = '#28a745';
            fetch('/control?action=start').then(r => r.text());
            updateData();
        }

        function stopAcquisition() {
            isAcquiring = false;
            document.getElementById('acquisitionStatus').textContent = 'Остановлено';
            document.getElementById('acquisitionStatus').style.color = '#dc3545';
            fetch('/control?action=stop').then(r => r.text());
        }

        function clearSpectrum() {
            spectrumChart.data.datasets[0].data = [];
            timeDomainChart.data.datasets[0].data = [];
            spectrogramData = [];
            spectrogramChart.data.datasets = [];
            spectrumChart.update();
            timeDomainChart.update();
            spectrogramChart.update();
        }

        function changeSampleRate() {
            const rate = document.getElementById('sampleRateSelect').value;
            fetch('/config?samplerate=' + rate).then(r => r.text());
        }

        function changeScale() {
            const scale = document.getElementById('scaleSelect').value;
            spectrumChart.options.scales.y.type = scale;
            spectrumChart.update();
        }

        // Обновление данных
        function updateData() {
            if (!isAcquiring) return;

            fetch('/spectrum')
                .then(response => response.json())
                .then(data => {
                    // Обновляем статистику
                    document.getElementById('sampleRate').textContent = data.sample_rate;
                    document.getElementById('bufferSize').textContent = data.buffer_size;
                    document.getElementById('updateRate').textContent = data.update_rate;
                    document.getElementById('maxAmplitude').textContent = data.max_amplitude.toFixed(1);

                    // Обновляем график спектра
                    const spectrumData = [];
                    const maxFreq = data.sample_rate / 2;

                    for (let i = 0; i < data.spectrum.length; i++) {
                        const freq = (i * maxFreq) / data.spectrum.length;
                        spectrumData.push({x: freq, y: data.spectrum[i]});
                    }

                    spectrumChart.data.datasets[0].data = spectrumData;
                    spectrumChart.update();

                    // Обновляем график временной области
                    const timeData = [];
                    for (let i = 0; i < Math.min(200, data.time_domain.length); i++) {
                        timeData.push({x: i, y: data.time_domain[i]});
                    }
                    timeDomainChart.data.datasets[0].data = timeData;
                    timeDomainChart.update();

                    // Обновляем спектрограмму
                    updateSpectrogram(data.spectrum);

                    // Следующее обновление
                    setTimeout(updateData, updateInterval);
                })
                .catch(error => {
                    console.error('Error:', error);
                    setTimeout(updateData, updateInterval);
                });
        }

        function updateSpectrogram(spectrum) {
            const timestamp = new Date().toLocaleTimeString();

            // Добавляем новые данные
            spectrogramData.push({
                label: timestamp,
                data: spectrum.map((val, idx) => ({x: idx, y: val})),
                borderColor: `hsl(${spectrogramData.length * 10}, 70%, 50%)`,
                borderWidth: 1,
                pointRadius: 0,
                fill: false
            });

            // Ограничиваем количество точек
            if (spectrogramData.length > maxSpectrogramPoints) {
                spectrogramData.shift();
            }

            // Обновляем график
            spectrogramChart.data.datasets = spectrogramData;
            spectrogramChart.update();
        }

        // Инициализация при загрузке
        document.addEventListener('DOMContentLoaded', function() {
            initializeCharts();
            // Автоматически запускаем приобретение данных
            startAcquisition();
        });
    </script>
</body>
</html>
)rawliteral";

// Обработчики веб-сервера
void handleRoot() {
    Serial.println("✅ GET / - Serving spectrum analyzer page");

    String html = FPSTR(INDEX_HTML);
    server.send(200, "text/html", html);
}

void handleSpectrum() {
    StaticJsonDocument<2048> doc;

    doc["sample_rate"] = SAMPLE_RATE;
    doc["buffer_size"] = BUFFER_SIZE;
    doc["update_rate"] = 1000 / updateInterval;

    // Находим максимальную амплитуду
    float maxAmp = -1000;
    for (int i = 0; i < SPECTRUM_BINS/2; i++) {
        if (spectrum[i] > maxAmp) maxAmp = spectrum[i];
    }
    doc["max_amplitude"] = maxAmp;

    // Добавляем спектр
    JsonArray spectrumArray = doc.createNestedArray("spectrum");
    for (int i = 0; i < SPECTRUM_BINS/2; i++) {
        spectrumArray.add(spectrum[i]);
    }

    // Добавляем временные данные (первые 200 точек)
    JsonArray timeArray = doc.createNestedArray("time_domain");
    portENTER_CRITICAL(&timerMux);
    for (int i = 0; i < min(200, BUFFER_SIZE); i++) {
        timeArray.add(adcBuffer[i]);
    }
    portEXIT_CRITICAL(&timerMux);

    String response;
    serializeJson(doc, response);
    server.send(200, "application/json", response);
}

void handleControl() {
    if (server.hasArg("action")) {
        String action = server.arg("action");
        Serial.println("🔧 Control command: " + action);

        if (action == "start") {
            // Перезапускаем таймер
            timerAlarmDisable(timer);
            sampleIndex = 0;
            bufferReady = false;
            timerAlarmEnable(timer);
            server.send(200, "text/plain", "Acquisition STARTED");
        } else if (action == "stop") {
            timerAlarmDisable(timer);
            server.send(200, "text/plain", "Acquisition STOPPED");
        } else {
            server.send(400, "text/plain", "Unknown command");
        }
    } else {
        server.send(400, "text/plain", "No command specified");
    }
}

void handleConfig() {
    if (server.hasArg("samplerate")) {
        int newRate = server.arg("samplerate").toInt();
        // Здесь можно изменить частоту дискретизации
        Serial.println("⚙️ Sample rate change requested: " + String(newRate));
        server.send(200, "text/plain", "Sample rate update received");
    }
}

void handleNotFound() {
    Serial.println("❌ 404 Not Found: " + server.uri());

    String message = "404: Not Found\n\n";
    message += "URI: ";
    message += server.uri();
    message += "\nMethod: ";
    message += (server.method() == HTTP_GET) ? "GET" : "POST";
    message += "\nArguments: ";
    message += String(server.args());
    message += "\n";

    for (uint8_t i = 0; i < server.args(); i++) {
        message += " ";
        message += server.argName(i);
        message += ": ";
        message += server.arg(i);
        message += "\n";
    }

    server.send(404, "text/plain", message);
}

void setupADC() {
    analogReadResolution(12); // 12-bit resolution
    analogSetAttenuation(ADC_11db); // 0-3.3V range

    // Настраиваем таймер для АЦП
    timer = timerBegin(0, 80, true); // 80 MHz / 80 = 1 MHz timer
    timerAttachInterrupt(timer, &onTimer, true);
    timerAlarmWrite(timer, 1000000 / SAMPLE_RATE, true); // Микросекунды
}

void setup() {
    Serial.begin(115200);
    delay(3000); // Даем время для открытия Serial Monitor

    Serial.println("\n" + String('=', 80));
    Serial.println("=== ESP32-C3 ADC Spectrum Analyzer ===");
    Serial.println(String('=', 80));

    // Настройка пина светодиода
    pinMode(LED_PIN, OUTPUT);
    digitalWrite(LED_PIN, LOW);

    // Настройка АЦП
    Serial.println("🔧 Setting up ADC...");
    setupADC();

    // Настройка WiFi
    Serial.println("📡 Starting WiFi Access Point...");

    WiFi.disconnect(true);
    delay(1000);
    WiFi.mode(WIFI_AP);

    bool apStatus = WiFi.softAP(WIFI_SSID, WIFI_PASSWORD, 1, 0, 4);
    delay(2000);

    Serial.println("✅ WiFi AP started: " + String(apStatus ? "SUCCESS" : "FAILED"));
    Serial.println("📶 Network Information:");
    Serial.println("  SSID: " + String(WIFI_SSID));
    Serial.println("  IP: " + WiFi.softAPIP().toString());
    Serial.println("  MAC: " + WiFi.softAPmacAddress());
    Serial.println("  Stations: " + String(WiFi.softAPgetStationNum()));

    // Настройка маршрутов сервера
    Serial.println("🌐 Setting up HTTP server routes...");

    server.on("/", handleRoot);
    server.on("/spectrum", handleSpectrum);
    server.on("/control", handleControl);
    server.on("/config", handleConfig);
    server.onNotFound(handleNotFound);

    server.begin();
    Serial.println("✅ HTTP server started on port " + String(WEB_SERVER_PORT));

    // Запускаем таймер АЦП
    timerAlarmEnable(timer);

    Serial.println("\n🎯 Spectrum Analyzer Ready!");
    Serial.println("   Connect to: http://" + WiFi.softAPIP().toString());
    Serial.println("   ADC Pin: GPIO" + String(ADC_PIN));
    Serial.println("   Sample Rate: " + String(SAMPLE_RATE) + " Hz");
    Serial.println("   Buffer Size: " + String(BUFFER_SIZE) + " samples");
    Serial.println(String('=', 80));

    // Индикация готовности
    digitalWrite(LED_PIN, HIGH);
    delay(500);
    digitalWrite(LED_PIN, LOW);
    delay(500);
    digitalWrite(LED_PIN, HIGH);
}

void loop() {
    server.handleClient();

    // Вычисляем спектр когда буфер готов
    if (bufferReady) {
        computeAutoSpectrum();
    }

    // Мигаем светодиодом для индикации работы
    static unsigned long lastBlink = 0;
    static bool ledState = false;

    if (millis() - lastBlink > 1000) {
        ledState = !ledState;
        digitalWrite(LED_PIN, ledState ? HIGH : LOW);
        lastBlink = millis();

        // Периодически выводим статус
        static unsigned long lastStatus = 0;
        if (millis() - lastStatus > 10000) {
            Serial.println("💡 Status - Uptime: " + String(millis() / 1000) +
                          "s, Clients: " + String(WiFi.softAPgetStationNum()) +
                          ", Free memory: " + String(esp_get_free_heap_size()) + " bytes");
            lastStatus = millis();
        }
    }

    delay(10);
}